 /**
 *
 * \file        HPRFGW_RFSlaveIntf.c
 *
 * \brief       This file contains the RF slave interface API functions. 		
 *
 * \detail		
 *
 * \author      Hazrat Shah	
 *
 * \date        11/28/2006
 * \warning	
 * \note
 * \review
 */
#include "hprfgw_config.h"
// MNT - 3/21/2007 - Dependency Removal tag begin
//#include "console.h"
//#include "hprfgw_eeprom.h"
// MNT - 3/21/2007 - Dependency Removal tag end
#include "hprfgw_rfResultCode.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfApp.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfRecMsg.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfDiagnostics.h"
#include "smac_app.h"

////////////////////////////////////////////////////////////////////////////////
//	EXTERNAL 	
////////////////////////////////////////////////////////////////////////////////
extern	INT8   *ConvertUIDtoString (UINT8 *puid, INT8 *pstr, UINT8 addrmode);
extern void SMAC_MCUPowerUp(void);
extern void SMAC_MCUPowerDown(void);
extern void RFReceive_TimerPowerUp(void);
extern void RFReceive_TimerPowerDown(void);

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS	
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA		
////////////////////////////////////////////////////////////////////////////////

void	RFAPPINTERFACE_InitRFApplication (void);
void	RFAPPINTERFACE_SetGatewayUID_PANID_ADDRMODE (BOOL Valid, UINT8 *pUID, UINT16 PanID, UINT8 AddrMode);	
void	RFAPPINTERFACE_SetLocalDeviceUID_PANID (BOOL Valid, UINT8 *pUID, UINT16 PanID);	
UINT16	RFAPPINTERFACE_PostCresnetPacket (UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive);
UINT16	RFAPPINTERFACE_DisconnectDeviceFromNetwork (void);
UINT16	RFAPPINTERFACE_ConnectDeviceToNetwork (UINT8 *pgatewayuid, UINT16 gatewaypanid, UINT8 channel, UINT8 gatewayaddrmode);
UINT16	RFAPPINTERFACE_LookupSettingsAndConnectDeviceToNetwork (void);
UINT16	RFAPPINTERFACE_EnterSleepMode (void);
UINT16	RFAPPINTERFACE_EnterPowerDownMode (void);
UINT16	RFAPPINTERFACE_PostLocalCresnetPacket (UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive);
BOOL	RFAPPINTERFACE_ControlSystemConnectionStatusQuery (void);
BOOL	RFAPPINTERFACE_GatewayConnectionStatusQuery (void);
UINT16  RFAPPINTERFACE_AcquireStart (UINT32 timeout);
UINT16	RFAPPINTERFACE_AcquireStop (BOOL ReStartNetwork);
BOOL	RFAPPINTERFACE_AcquireStatus (void);
void	RFAPPINTERFACE_SetRetryCount  (UINT8 RetryCnt);
void	RFAPPINTERFACE_SetRetryDelay (UINT8 *pRetryDly, UINT8 Cnt, UINT8 Index);
void	RFAPPINTERFACE_SetPowerLevel (UINT16 PowerLevel);
void	RFAPPINTERFACE_SetChannel (UINT8 Channel);	
void	RFAPPINTERFACE_SetCCAEnergyDetectThr (UINT8 EnergyDetectThr);
void	RFAPPINTERFACE_SetRSSIPowerCompOffset (UINT8 RSSIPowerCompOffset );
void	RFAPPINTERFACE_SetResultCodeTimeout (UINT16 ResultCodeTimeout);
void	RFAPPINTERFACE_SetIDString (UINT8 rfid, UINT32 tsid, UINT8 *pIDStr);
UINT16	RFAPPINTERFACE_ResetRFTransmitBuffers (void);
INT32	RFAPPINTERFACE_SetDeviceDataField (UINT8 field, INT8 *pbuf, UINT16 len);
INT32	RFAPPINTERFACE_PrepareAndPostDeviceDataInfoStr (UINT8 field);
INT32	RFAPPINTERFACE_PostEchoPacketRequest (TYPE_RF_ECHO_REQUEST_PACKET *pEchoRequest);
UINT16	RFAPPINTERFACE_ConvertLQIToIEEEScaleWord (UINT8 LQI, INT8 offset);	
INT32	RFAPPINTERFACE_ConvertLQITodBm (UINT8 LQI);
INT32	RFAPPINTERFACE_GetLQI (UINT8 RFID, BOOL LocalDevice, UINT8 *plinkquality);
UINT16	RFAPPINTERFACE_GetLocalDevicePower (void);
float	RFAPPINTERFACE_ConvertPowerTodBm (UINT16 power, BOOL Translate);
INT32	RFAPPINTERFACE_GoingToStandBy (void);
INT32	RFAPPINTERFACE_ComingoutofStandBy (void);
INT32	RFAPPINTERFACE_NetworkService (TYPE_RF_DISCOVER_REQUEST *pdisc);
INT32	RFAPPINTERFACE_SetRetryCountOnConnectionReq  (UINT8 RetryCnt);
INT32	RFAPPINTERFACE_SetPreLinkQueryPacketCnt (UINT8 Cnt);
INT32	RFAPPINTERFACE_PostCresnetPacketByRFID (UINT8 rfid, UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive);
INT32	RFAPPINTERFACE_PostLocalCresnetPacketByRFID (UINT8 rfid, UINT8 *pBuf, UINT16 BufLen, UINT8 priority, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive);
UINT16	RFAPPINTERFACE_PowerDown (void);
UINT16	RFAPPINTERFACE_PowerUp (void);

void RFAPP_ClearDiagCounters(void)
{
    Rfdiags_clearNumRxInactivityResets();
    Smac_clearCounters();
}


const TYPE_INTERFACE_TO_RFAPP_APP RFIntRFApp = {
	RFAPPINTERFACE_InitRFApplication,
	RFAPPINTERFACE_SetGatewayUID_PANID_ADDRMODE,
	RFAPPINTERFACE_SetLocalDeviceUID_PANID,
	RFAPPINTERFACE_PostCresnetPacket,
	RFAPPINTERFACE_DisconnectDeviceFromNetwork,
	RFAPPINTERFACE_ConnectDeviceToNetwork,
	RFAPPINTERFACE_LookupSettingsAndConnectDeviceToNetwork,
	RFAPPINTERFACE_EnterSleepMode,
	RFAPPINTERFACE_EnterPowerDownMode,
	RFAPPINTERFACE_PostLocalCresnetPacket,
	RFAPPINTERFACE_ControlSystemConnectionStatusQuery,
	RFAPPINTERFACE_GatewayConnectionStatusQuery,	
	RFAPPINTERFACE_AcquireStart,
	RFAPPINTERFACE_AcquireStop,
	RFAPPINTERFACE_AcquireStatus,
	RFAPPINTERFACE_SetRetryCount,
	RFAPPINTERFACE_SetRetryDelay,
	RFAPPINTERFACE_SetPowerLevel,
	RFAPPINTERFACE_SetChannel,
	RFAPPINTERFACE_SetCCAEnergyDetectThr,
	RFAPPINTERFACE_SetRSSIPowerCompOffset,
	RFAPPINTERFACE_SetResultCodeTimeout,
	RFAPPINTERFACE_SetIDString,
	RFAPPINTERFACE_ResetRFTransmitBuffers,
	RFAPPINTERFACE_SetDeviceDataField,
	RFAPPINTERFACE_PrepareAndPostDeviceDataInfoStr,
	RFAPPINTERFACE_PostEchoPacketRequest,	
	RFAPPINTERFACE_ConvertLQIToIEEEScaleWord,	
	RFAPPINTERFACE_ConvertLQITodBm,
	RFAPPINTERFACE_GetLQI,
	RFAPPINTERFACE_ConvertPowerTodBm, 
	RFAPPINTERFACE_GetLocalDevicePower,
	RFAPPINTERFACE_GoingToStandBy,	
	RFAPPINTERFACE_ComingoutofStandBy,
	RFAPPINTERFACE_NetworkService,
	RFAPPINTERFACE_SetRetryCountOnConnectionReq,
	RFAPPINTERFACE_SetPreLinkQueryPacketCnt,	
	RFAPPINTERFACE_PostCresnetPacketByRFID,
	RFAPPINTERFACE_PostLocalCresnetPacketByRFID,
	RFAPPINTERFACE_PowerDown,
	RFAPPINTERFACE_PowerUp
};

////////////////////////////////////////////////////////////////////////////////
//				
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines executes the rf initialization call  	 			           
 *
 * \detail		
 * \date        11/28/2006
 * \param	    none		
 *
 * \return		none
 *
 * \retval				
 */
void	RFAPPINTERFACE_InitRFApplication (void)	{
	if (SMAC_IsRFChipValid() == TRUE)
    {
        RFIntHostApp.pF_ErrorPrintf("RF PHY Valid (Chip Id: 0x%x)\n", RF_Intfc_GetVersion());
	    RFReceive_Init();
	    RFTransmit_Init();
	    RFAPPInit();
    } // if (SMAC_IsRFChipValid() == TRUE)
  else
    {
    	RFIntHostApp.pF_ErrorPrintf("RF PHY Invalid\n");     
    } // if (SMAC_IsRFChipValid() == TRUE)
}

void	RFAPPINTERFACE_SetGatewayUID_PANID_ADDRMODE (BOOL Valid, UINT8 *pUID, UINT16 PanID, UINT8 AddrMode)	{
	//RFReceive_SetGatewayUIDValid (Valid);	
	RFDesc_SetUID	(DEVICE_DESCRIPTIVE_TABLE_INDEX, pUID);
	RFDesc_SetPanID (DEVICE_DESCRIPTIVE_TABLE_INDEX, PanID);	
	RFDesc_SetAddrMode (DEVICE_DESCRIPTIVE_TABLE_INDEX, AddrMode);
}

void	RFAPPINTERFACE_SetLocalDeviceUID_PANID (BOOL Valid, UINT8 *pUID, UINT16 PanID)	{
	RFReceive_SetLocalDeviceUIDValid (Valid);
	RFReceive_SetLocalDeviceUID (pUID);
	RFReceive_SetLocalDevicePanID (PanID);
}

UINT16	RFAPPINTERFACE_PostCresnetPacket (UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
//#ifdef	RF_NETWORK_MASTER_DEVICE	
//	return (RFTransmit_BuildCresnetPacketAndPostToPipeByRFID (*pBuf, pBuf, BufLen, pcbf, cbfpar, timetolive));
//#else
	return (RFTransmit_BuildCresnetPacketAndPostToPipeByIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX, pBuf, BufLen, pcbf, cbfpar, timetolive));
//#endif
}

UINT16	RFAPPINTERFACE_DisconnectDeviceFromNetwork (void)	{
	return (FAILURE);
}

UINT16	RFAPPINTERFACE_ConnectDeviceToNetwork (UINT8 *pgatewayuid, UINT16 gatewaypanid, UINT8 channel, UINT8 addrmode)	{

#ifdef	RF_NETWORK_MASTER_DEVICE
	return (FAILURE);
#else
	return (RFAPP_EstablishConnectionwithGateway(pgatewayuid, gatewaypanid, channel, addrmode));	
#endif
}

UINT16	RFAPPINTERFACE_LookupSettingsAndConnectDeviceToNetwork (void)	{	
  BOOL	gatewayuidvalid;
	UINT8	gatewayuid[UID_MAX_SIZE], channel;
	UINT16	gatewaypanid;
	UINT8	gatewayaddrmode;	
	
	RFIntHostApp.pF_GetChannel	(&channel, SLAVE_DEVICE_DEFAULT_CHANNEL);
	RFIntHostApp.pF_GetGatewayUID_PANID_ADDRMODE(&gatewayuidvalid, gatewayuid, &gatewaypanid, &gatewayaddrmode, (UINT8 *)GATEWAY_DEFAULT_UID, GATEWAY_DEFAULT_PANID, GATEWAY_ADDR_MODE);
	RFAPPINTERFACE_ConnectDeviceToNetwork (gatewayuid, gatewaypanid, channel, gatewayaddrmode);	

	return (SUCCESS);
}

UINT16	RFAPPINTERFACE_EnterSleepMode (void)	{
	return (RFTransmit_BuildSleepModePacketAndPostToPipeByIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX));
}

UINT16	RFAPPINTERFACE_EnterPowerDownMode (void)	{
  return (RFTransmit_BuildPowerDownModePacketAndPostToPipeByIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX));
}

UINT16	RFAPPINTERFACE_PostLocalCresnetPacket (UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
//#ifdef	RF_NETWORK_MASTER_DEVICE
//	return (RFTransmit_BuildLocalCresnetPacketAndPostToPipeByRFID (*pBuf, pBuf, BufLen, pcbf, cbfpar, timetolive));	
//#else
	return (RFTransmit_BuildLocalCresnetPacketAndPostToPipeByIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX, pBuf, BufLen, pcbf, cbfpar, timetolive));	
//#endif
 }

BOOL	RFAPPINTERFACE_ControlSystemConnectionStatusQuery (void)	{
  return (RFReceive_IsControlSystemConnected());	
}

BOOL	RFAPPINTERFACE_GatewayConnectionStatusQuery (void)	{
  return (RFReceive_IsGatewayConnected());	
}

UINT16	RFAPPINTERFACE_AcquireStart (UINT32 timeout)	{
  return(RFAPP_AcquireStart(timeout));
}

UINT16	RFAPPINTERFACE_AcquireStop (BOOL ReStartNetwork)	{
	return(RFAPP_AcquireStop(ReStartNetwork));		
}

BOOL	RFAPPINTERFACE_AcquireStatus (void)	{
	return (RFAPP_AcquireStatus());
}

void	RFAPPINTERFACE_SetRetryCount  (UINT8 RetryCnt)	{
	RFReceive_SetLocalDeviceRetry (RetryCnt);	
}

void	RFAPPINTERFACE_SetRetryDelay (UINT8 *pRetryDly, UINT8 Cnt, UINT8 Index)	{
	RFReceive_SetLocalDeviceRetryDelay (pRetryDly, Cnt, Index);
}

void	RFAPPINTERFACE_SetPowerLevel (UINT16 PowerLevel)	{
	RFReceive_SetLocalDevicePower (PowerLevel);
}

void	RFAPPINTERFACE_SetChannel (UINT8 Channel)	{	
	RFReceive_SetLocalDeviceChannel (Channel);	
}

void	RFAPPINTERFACE_SetCCAEnergyDetectThr (UINT8 CCAEnergyDetectThr)	{
	RFReceive_SetLocalDeviceCCAEnergyDetectThr (CCAEnergyDetectThr);
}

void	RFAPPINTERFACE_SetRSSIPowerCompOffset (UINT8 RSSIPowerCompOffset)	{
	RFReceive_SetLocalDeviceRSSIPowerCompOffset (RSSIPowerCompOffset);
}

void	RFAPPINTERFACE_SetResultCodeTimeout (UINT16 ResultCodeTimeout)	{
	RFReceive_SetLocalDeviceResultCodeTimeout (ResultCodeTimeout);
}

void	RFAPPINTERFACE_SetIDString (UINT8 rfid, UINT32 tsid, UINT8 *pIDStr)	{
	//	only used for reporting id string change (not to be used as a reply to the CRESNET id request command)	
	//	RFTransmit_BuildIDStringPacketAndPostToPipeByIndex (DEVICE_DESCRIPTIVE_TABLE_INDEX, rfid, tsid, pIDStr);	
}

UINT16	RFAPPINTERFACE_ResetRFTransmitBuffers (void)	{
	UINT16 res = SUCCESS;

	//	This function is used from the host application to reset the RF applicaiton transmit buffers

	res = RFTransmit_ResetTxBuffers (DEVICE_DESCRIPTIVE_TABLE_INDEX);

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines copies the device field information to the stack buffer
 *				and reports the change to the Ctl system
 *
 * \detail		
 * \date        07/09/07
 *
 * \param	    UINT8 field, INT8 *pbuf, UINT16 len		
 *
 * \return		INT32
 *
 * \retval				
 */
INT32	RFAPPINTERFACE_SetDeviceDataField (UINT8 field, INT8 *pbuf, UINT16 len)	{
	return (RFAPP_DeviceDataInfoStrChange(field, pbuf, len));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine is called to post the device info string to the gateway
 *
 * \detail		
 * \date        07/10/07
 *
 * \param	    UINT8 field	
 *
 * \return		INT32
 *
 * \retval				
 */
INT32	RFAPPINTERFACE_PrepareAndPostDeviceDataInfoStr (UINT8 field)	{
	return (RFAPP_PrepareAndPostDeviceDataInfoStr(field));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine is called to start the echo test  
 *
 * \detail		
 * \date        07/10/07
 *
 * \param	    TYPE_RF_ECHO_REQUEST_PACKET *pEchoRequest	
 *
 * \return		INT32
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFAPPINTERFACE_PostEchoPacketRequest (TYPE_RF_ECHO_REQUEST_PACKET *pEchoRequest)	{
	return (RFAPP_PostEchoRequest(pEchoRequest));	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine converts the passed on LQI value to IEEE recommended converted value   
 *
 * \detail		
 * \date        07/10/07
 *
 * \param	    UINT8 LQI, INT8 offset	
 *
 * \return		UINT16
 *
 * \retval		converted LQI		
 */
UINT16	RFAPPINTERFACE_ConvertLQIToIEEEScaleWord (UINT8 LQI, INT8 offset)	{
	return (RFReceive_ConvertLinkQualityToIEEEWord(LQI, offset));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine converts the passed on LQI value to IEEE recommended converted value   
 *
 * \detail		
 * \date        07/10/07
 *
 * \param	    UINT8 LQI	
 *
 * \return		INT32
 *
 * \retval		converted LQI		
 */
INT32	RFAPPINTERFACE_ConvertLQITodBm	(UINT8 LQI)	{
	return ((INT32)RFReceive_ConvertLinkQualityTodBm(LQI));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine returns the local device or remote device link quality 	   
 *
 * \detail		
 * \date        07/10/07
 *
 * \param	    UINT8 RFID, BOOL LocalDevice, UINT8 *linkquality (TRUE=Local device, FALSE=remote device)	
 *
 * \return		INT32 res
 *
 * \retval		SUCCESS/FAILURE	
 */
INT32	RFAPPINTERFACE_GetLQI (UINT8 RFID, BOOL LocalDevice, UINT8 *plinkquality)	{
	UINT8	index;
#ifdef	RF_NETWORK_MASTER_DEVICE
	UINT8	indexarray[MAX_DESCRIPTIVE_TABLE_ENTRIES+2];
#endif
	INT32	res = FAILURE;

	*plinkquality = 255;

#ifdef	RF_NETWORK_MASTER_DEVICE
	if (RFDesc_GetActiveRFDeviceTableIndexByRFID (RFID, indexarray))	{
		index = indexarray[0];		
#else 
	{
		index = DEVICE_DESCRIPTIVE_TABLE_INDEX;
#endif	
		if (LocalDevice)	{
			RFDesc_GetLinkQualityOfLocalDevice(index, plinkquality);	
		}	else	{
			RFDesc_GetLinkQualityOfRemoteDevice(index, plinkquality);
		}
		res = SUCCESS;
	}
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine converts the power level to dBm value  	   
 *
 * \detail		
 * \date        07/10/07
 *
 * \param	    UINT16 power, BOOL Translate	
 *
 * \return		float 
 *
 * \retval		power level in dBm	
 */
float	RFAPPINTERFACE_ConvertPowerTodBm (UINT16 power, BOOL Translate)	{
  return(RFReceive_ConvertPowerTodBm(power, Translate));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine returns the power level index stored in rf stack 	   
 *
 * \detail		
 * \date        07/22/07
 *
 * \param	    void		
 *
 * \return		UINT16
 *
 * \retval		power level index (High/Low) 	
 */
UINT16	RFAPPINTERFACE_GetLocalDevicePower (void)	{
  return (RFReceive_GetLocalDevicePower());
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine is called when the host device goes into standby mode  	   
 *
 * \detail		
 * \date        09/25/07
 *
 * \param	    void		
 *
 * \return		INT32
 *
 * \retval		SUCCESS/FAILURE	 	
 */
INT32	RFAPPINTERFACE_GoingToStandBy (void)	{
#ifndef	RF_NETWORK_MASTER_DEVICE
	//	if device is not linked, stop the link process to save battery
	//	the link process then restarts when the device comes out of standby mode
	if (!RFReceive_IsGatewayConnected())	{
		RFAPP_LinkRequestHandlerStop();
	}
#endif
	return (SUCCESS);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine is called when the host device comes out of standby mode  	   
 *
 * \detail		
 * \date        09/25/07
 *
 * \param	    void		
 *
 * \return		INT32
 *
 * \retval		SUCCESS/FAILURE	 	
 */
INT32	RFAPPINTERFACE_ComingoutofStandBy (void)	{
#ifndef	RF_NETWORK_MASTER_DEVICE	
	//	if device is not linked, start the link process immediatly 
	if (!RFReceive_IsGatewayConnected())	{
		RFAPP_LinkRequestHandlerStart(0);
	}
#endif
	return (SUCCESS);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine scans for gateways (could be acquired or not acquired)  	   
 *				
 * \detail		One of the purpose of this function is to test the rf receiver/transmitter
 *				without the device being acquired by the gateway typically used during manufacturing
 *				
 * \date        10/02/07
 *
 * \param	    TYPE_RF_DISCOVER_REQUEST *pdisc		
 *
 * \return		INT32
 *
 * \retval		SUCCESS/FAILURE	 	
 */
INT32	RFAPPINTERFACE_NetworkService (TYPE_RF_DISCOVER_REQUEST *pdisc)	{
  return (RFAPP_RFNetworkDiscover(pdisc));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine updates the local image with the latest retry count   	   
 *				
 * \detail		
 *				
 *				
 * \date        10/30/07
 *
 * \param	    UINT8 RetryCnt		
 *
 * \return		INT32
 *
 * \retval
 */
INT32	RFAPPINTERFACE_SetRetryCountOnConnectionReq (UINT8 RetryCnt)	{
	return ((INT32)RFReceive_SetLocalDeviceRetryOnConnectionReq (RetryCnt));	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine updates the prelink query packet count.
 *				The prelink query packets are sent prior to sending the link request
 *				   	   
 *				
 * \detail		
 *				
 *				
 * \date        11/02/07
 *
 * \param	    UINT8 Cnt		
 *
 * \return		INT32
 *
 * \retval
 */
INT32	RFAPPINTERFACE_SetPreLinkQueryPacketCnt (UINT8 Cnt)	{
	return ((INT32)RFReceive_SetLocalDevicePreLinkQueryPacketCnt (Cnt));	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine posts the cresnet packet to the rf transmit task 
 *				   	   
 *				
 * \detail		
 *				
 *				
 * \date        11/07/07
 *
 * \param	    UINT8 rfid, UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive		
 *
 * \return		INT32
 *
 * \retval
 */
INT32	RFAPPINTERFACE_PostCresnetPacketByRFID (UINT8 rfid, UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
  return ((INT32)RFTransmit_BuildCresnetPacketAndPostToPipeByRFID (rfid, pBuf, BufLen, pcbf, cbfpar, timetolive));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routine posts the local cresnet packet to the rf transmit task 
 *				   	   
 *				
 * \detail		
 *				
 *				
 * \date        11/07/07
 *
 * \param	    UINT8 rfid, UINT8 *pBuf, UINT16 BufLen, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive		
 *
 * \return		INT32
 *
 * \retval
 */
INT32	RFAPPINTERFACE_PostLocalCresnetPacketByRFID (UINT8 rfid, UINT8 *pBuf, UINT16 BufLen, UINT8 priority, INT32(*pcbf)(UINT32 par, UINT32 status), UINT32 cbfpar, UINT32 timetolive)	{
  return ((INT32)RFTransmit_BuildLocalCresnetPacketAndPostToPipeByRFID (rfid, pBuf, BufLen, priority, pcbf, cbfpar, timetolive));
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief       Power down sequence for RF module 
 *				   	   
 *				
 * \detail
 *				
 *				
 * \date        11/07/07
 *
 * \param       None
 *
 * \return      INT32
 *
 * \retval
 */
UINT16 RFAPPINTERFACE_PowerDown (void)
{
    // Set the SMAC to power down state
    SMAC_MCUPowerDown();

    // Power down the Receive timer
    RFReceive_TimerPowerDown();

    return TRUE;
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief       Power up sequence for RF module 
 *				   	   
 *				
 * \detail
 *				
 *				
 * \date        11/07/07
 *
 * \param       None
 *
 * \return      INT32
 *
 * \retval
 */
UINT16 RFAPPINTERFACE_PowerUp (void)
{
    // Power up the receive timer
    RFReceive_TimerPowerUp();

    // SMAC power down function
    SMAC_MCUPowerUp();
    
    return TRUE;
}
